const moment = require('moment-timezone');
const db = require('../database/models');
const utils = require('../utils/utils');
const serviceUtils = require('./utils.js');

const {
  Habit, UserHabit, UserAspectReinventation, UserNotification,
} = db;

exports.getNextWeeekdayDate = (dayId, timeZone) => {
  const now = moment.tz(timeZone);
  let day = moment.tz(timeZone).isoWeekday(dayId);
  if (day.diff(now, 'days') < 1) {
    day = moment.tz(timeZone).isoWeekday(dayId).add(7, 'days');
  }
  return day;
};

exports.getHabits = async userId => {
  try {
    const currentRoute = await serviceUtils.getCurrentUserRoute(userId);
    const { id: routeId = '' } = currentRoute;

    const currentChallenge = await serviceUtils.getCurrentUserChallenge(userId, routeId);
    const { id: challengeId = '' } = currentChallenge;

    const userAspectReinventationId = await serviceUtils.getCurrentUserAspectReinventation(
      userId,
      challengeId,
      5,
    );

    const habits = await Habit.findAll({
      attributes: ['id', 'name'],
      where: { challengeId },
      include: [
        {
          required: false,
          model: UserHabit,
          as: 'userHabit',
          attributes: ['id', 'active', 'completedDate', 'days', 'hour'],
          where: {
            userAspectReinventationId,
          },
        },
      ],
      order: [['userHabit', 'completedDate'], ['userHabit', 'active', 'desc']],
    });

    const finalResponse = [];
    habits.forEach(h => {
      const { id = '', name = '', userHabit = [] } = h;
      const currentUserHabit = userHabit[0] || [];
      finalResponse.push({
        id,
        name,
        // userHabitId: currentUserHabit.id || null,
        active: currentUserHabit.active || 0,
        completed: currentUserHabit.completedDate || 0,
        days: currentUserHabit.days || [],
        hour: currentUserHabit.hour || '',
      });

    });

    return {
      habits: finalResponse,
      currentRoute,
      currentChallenge,
    };
  } catch (e) {
    throw e;
  }
};

exports.activeUserHabit = async ({
  userId,
  habitId,
  days,
  hour,
}) => {
  try {
    const { id: routeId = '' } = await serviceUtils.getCurrentUserRoute(userId);

    const { id: challengeId = '' } = await serviceUtils.getCurrentUserChallenge(userId, routeId);
    const userAspectReinventationId = await serviceUtils.getCurrentUserAspectReinventation(
      userId,
      challengeId,
      5,
    );

    const active = days.length ? 1 : 0;

    await utils.updateOrCreate(
      UserHabit,
      { habitId, userAspectReinventationId },
      {
        habitId, userAspectReinventationId, active, days, hour,
      },
    );

    await UserNotification.destroy({
      where: {
        userId,
        habitId,
        sent: null,
      },
    });

    await exports.createHabitWeekNotifications({
      userId,
      habitId,
    });

    return {};
  } catch (e) {
    throw e;
  }
};

exports.completeUserHabit = async ({ userId, habitId, completed }) => {
  try {
    const { id: routeId = '' } = await serviceUtils.getCurrentUserRoute(userId);
    const { id: challengeId = '' } = await serviceUtils.getCurrentUserChallenge(userId, routeId);
    const userAspectReinventationId = await serviceUtils.getCurrentUserAspectReinventation(
      userId,
      challengeId,
      5,
    );

    await utils.updateOrCreate(
      UserHabit,
      { habitId, userAspectReinventationId },
      { habitId, userAspectReinventationId, completedDate: completed ? moment().utc() : null },
    );

    if (completed) {
      await UserNotification.destroy({
        where: {
          userId,
          habitId,
          sent: null,
        },
      });
    } else {
      await exports.createHabitWeekNotifications({
        userId,
        habitId,
      });
    }

    return {};
  } catch (e) {
    throw e;
  }
};

exports.setActiveHabits = async userId => {
  try {
    const { id: routeId = '' } = await serviceUtils.getCurrentUserRoute(userId);

    const { id: challengeId = '' } = await serviceUtils.getCurrentUserChallenge(userId, routeId);

    const userAspectReinventationId = await serviceUtils.getCurrentUserAspectReinventation(
      userId,
      challengeId,
      5,
    );

    await UserAspectReinventation.update(
      { isActive: 1 },
      { where: { id: userAspectReinventationId } },
    );

    await serviceUtils.addToUserAspectReinventation(userId, challengeId, 5);
    return {};
  } catch (e) {
    throw e;
  }
};

exports.createHabitWeekNotifications = async ({ userId, habitId }) => {
  const timeZone = await serviceUtils.getUserTimeZone(userId);

  const pendingUserHabits = await UserHabit.findAll({
    include: [
      {
        required: true,
        model: UserAspectReinventation,
        as: 'userAspectReinventation',
        attributes: ['userId'],
        where: {
          userId,
        },
      },
      {
        required: true,
        model: Habit,
        as: 'habit',
        attributes: ['id', 'name'],
      },
    ],
    where: {
      completedDate: null,
      habitId,
    },
  });

  const dates = {
    LUNES: exports.getNextWeeekdayDate(1, timeZone),
    MARTES: exports.getNextWeeekdayDate(2, timeZone),
    MIERCOLES: exports.getNextWeeekdayDate(3, timeZone),
    JUEVES: exports.getNextWeeekdayDate(4, timeZone),
    VIERNES: exports.getNextWeeekdayDate(5, timeZone),
    SABADO: exports.getNextWeeekdayDate(6, timeZone),
    DOMINGO: exports.getNextWeeekdayDate(0, timeZone),
  };

  console.log('createHabitWeekNotifications dates', JSON.stringify(dates, null, 2));

  const notificationsToCreate = [];

  pendingUserHabits.forEach(userHabit => {
    const {
      days = [], userAspectReinventation = {}, habit = '', hour: hourSelected,
    } = userHabit;
    const [hour, minute] = hourSelected.split(':');
    const { id, name } = habit;

    days.forEach(d => {
      const notificationToCreate = {
        userId: userAspectReinventation.userId,
        habitId: id,
        date: dates[d].set({
          hour: parseInt(hour), minute: parseInt(minute),
        }),
        title: `💡 BeCrackHábitos: tiempo de ${name}`,
        body: `Tiempo de ${name}`,
      };

      console.log('Habit notificationToCreate', JSON.stringify(notificationToCreate, null, 2));

      notificationsToCreate.push({
        userId: userAspectReinventation.userId,
        habitId: id,
        date: dates[d].set({
          hour: parseInt(hour), minute: parseInt(minute),
        }),
        title: `💡 BeCrackHábitos: tiempo de ${name}`,
        body: `Tiempo de ${name}`,
      });
    });
  });

  if (notificationsToCreate.length) {
    await UserNotification.bulkCreate(notificationsToCreate);
  }

};
