const Aigle = require('aigle');
const _ = require('lodash');
const db = require('../database/models');
const utils = require('../utils/utils');
const serviceUtils = require('./utils.js');
const aws = require('../aws');

const {
  PowerfulQuestion, UserPowerfulQuestion, UserAspectReinventation, Challenge,
} = db;

exports.getPowerfulQuestions = async userId => {
  try {
    const currentRoute = await serviceUtils.getCurrentUserRoute(userId);
    const { id: routeId = '' } = currentRoute;

    const currentChallenge = await serviceUtils.getCurrentUserChallenge(userId, routeId);
    const { id: challengeId = '' } = currentChallenge;

    const challenge = await Challenge.findOne({ where: { id: challengeId } });

    const userAspectReinventationId = await serviceUtils.getCurrentUserAspectReinventation(
      userId,
      challengeId,
      1,
    );

    const powerfulQuestionsResult = await PowerfulQuestion.findAll({
      attributes: ['id', 'name', 'subtitle', 'description'],
      where: { challengeId },
      include: [
        {
          required: false,
          model: UserPowerfulQuestion,
          as: 'userPowerfulQuestion',
          attributes: ['id', 'value'],
          where: {
            userAspectReinventationId,
          },
        },
      ],
    });

    let powerfulQuestions = [];
    powerfulQuestionsResult.forEach(h => {
      const {
        id = '', name = '', userPowerfulQuestion = [], subtitle,
        description,
      } = h;
      const currentUserPowerfulQuestion = userPowerfulQuestion[0] || [];
      powerfulQuestions.push({
        id,
        name,
        subtitle,
        description,
        value: currentUserPowerfulQuestion.value || 0,
      });
    });

    powerfulQuestions = _.orderBy(powerfulQuestions, ['value'], ['desc']);

    const isLink = challenge.bitacoraLink && challenge.bitacoraLink.includes('https');

    const bitacoraLink = isLink ? challenge.bitacoraLink : aws.getSignedUrl(challenge.bitacoraLink);

    return {
      powerfulQuestions,
      bitacoraLink,
      currentChallenge,
      currentRoute,
    };
  } catch (e) {
    throw e;
  }
};

exports.updateValuesPowerfulQuestions = async (powerfulQuestionsIds, userId) => {
  try {
    const { id: routeId = '' } = await serviceUtils.getCurrentUserRoute(userId);

    const { id: challengeId = '' } = await serviceUtils.getCurrentUserChallenge(userId, routeId);

    const userAspectReinventationId = await serviceUtils.getCurrentUserAspectReinventation(
      userId,
      challengeId,
      1,
    );

    const hasUserPowerfullQuestion = await UserPowerfulQuestion.findAll({
      where: { userAspectReinventationId },
    });

    if (!hasUserPowerfullQuestion.length) {
      await serviceUtils.addToUserAspectReinventation(userId, challengeId, 1);
      await UserAspectReinventation.update(
        { isActive: 1 },
        { where: { id: userAspectReinventationId } },
      );
    }

    await Aigle.mapSeries(powerfulQuestionsIds, async pq => {
      const { id: powerfulQuestionId = '', value = '' } = pq;
      await utils.updateOrCreate(
        UserPowerfulQuestion,
        { powerfulQuestionId, userAspectReinventationId },
        { powerfulQuestionId, userAspectReinventationId, value: value || 0 },
      );
    });

    return {};
  } catch (e) {
    throw e;
  }
};
