const Aigle = require('aigle');
const _ = require('lodash');
const db = require('../database/models');
const utils = require('../utils/utils');

const {
  Route,
  Challenge,
  AspectReinventation,
  AspectReinventationType,
  UserAspectReinventation,
  User,
  Country,
} = db;

exports.getCurrentUserRoute = async userId => {
  try {
    const currentRoute = Route.findOne({
      include: [
        {
          required: true,
          model: User,
          as: 'user',
          attributes: [],
          through: {
            attributes: [],
            where: {
              disabledDate: null,
            },
          },
          where: {
            id: userId,
          },
        },
      ],
    });

    if (!currentRoute) {
      const error = {
        name: 'USUARIO_SIN_RUTA',
      };
      throw error;
    }

    return currentRoute;
  } catch (e) {
    throw e;
  }
};

exports.getCurrentUserChallenge = async (userId, routeId) => {
  try {
    const routeChallenges = await Challenge.findAll({ where: { routeId } });

    const challengeIds = routeChallenges.map(c => c.id);

    const currentChallenge = await Challenge.findOne({
      include: [
        {
          required: true,
          model: User,
          as: 'userChallenges',
          attributes: [],
          through: {
            attributes: [],
            where: {
              challengeId: challengeIds,
              current: 1,
            },
          },
          where: {
            id: userId,
          },
        },
      ],
    });

    return currentChallenge;
  } catch (e) {
    throw e;
  }
};

exports.getCurrentUserAspectReinventation = async (
  userId,
  challengeId,
  aspectReinventationTypeId,
) => {
  try {
    const aspectReinventation = await AspectReinventation.findAll({
      where: { challengeId, aspectReinventationTypeId },
    });

    const aspectReinventationId = aspectReinventation.length ? aspectReinventation[0].id : '';

    const currentUserAspectReinventationId = await UserAspectReinventation.findOne({
      where: { aspectReinventationId, userId },
    });

    const { id: userAspectReinventationId = '' } = currentUserAspectReinventationId;
    return userAspectReinventationId;
  } catch (e) {
    throw e;
  }
};

exports.addToUserAspectReinventation = async (userId, challengeId, blockingId) => {
  try {
    let unlockingIds = await AspectReinventationType.findAll({ where: { blockingId } });

    unlockingIds = unlockingIds.map(c => c.id);

    const addToUserAspectReinventation = await AspectReinventation.findAll({
      where: { aspectReinventationTypeId: unlockingIds, challengeId },
    });

    await Aigle.mapSeries(addToUserAspectReinventation, async x => {
      const { id: auxUserAspectReinventationId } = x;

      const existUAR = await UserAspectReinventation.findOne({
        where: { aspectReinventationId: auxUserAspectReinventationId, userId },
      });

      const isActive = existUAR && !existUAR.isActive ? 1 : 0;

      if (!existUAR || !existUAR.isActive) {
        await utils.updateOrCreate(
          UserAspectReinventation,
          { aspectReinventationId: auxUserAspectReinventationId, userId },
          { aspectReinventationId: auxUserAspectReinventationId, userId, isActive },
        );
      }

    });

    return {};
  } catch (e) {
    throw e;
  }
};

exports.getUserTimeZone = async userId => {
  try {
    const user = await User.findOne({
      where: {
        id: userId,
      },
      include: [
        {
          required: true,
          model: Country,
          as: 'country',
          attributes: ['timeZone'],
        },
      ],
    });

    const timezone = _.get(user, 'country.timeZone', 'UTC');

    return timezone;
  } catch (e) {
    throw e;
  }
};

exports.getRandomInt = max => Math.floor(Math.random() * max);
