const axios = require('axios');
const _ = require('lodash');
const PdfPrinter = require('pdfmake');
const fs = require('fs');
const moment = require('moment');
const path = require('path');
const CTS = require('./constants');
const config = require('../config');
const awsActions = require('../aws');
const pdfTexts = require('./pdfTexts');
const appText = require('./appTexts');

exports.loadDataFile = fileName => {
  try {
    let resolved;

    const lambdaPath = path.resolve(process.env.LAMBDA_TASK_ROOT, `utils/${fileName}`);

    if (fs.existsSync(lambdaPath)) {
      resolved = lambdaPath;
    } else {
      resolved = path.resolve(__dirname, fileName);
    }

    const data = fs.readFileSync(resolved);
    return data;
  } catch (error) {
    console.log(error.message);
  }
};

const fonts = {
  Roboto: {
    normal: exports.loadDataFile('fonts/Roboto/Roboto-Regular.ttf'),
    bold: exports.loadDataFile('fonts/Roboto/Roboto-Medium.ttf'),
    italics: exports.loadDataFile('fonts/Roboto/Roboto-Italic.ttf'),
    bolditalics: exports.loadDataFile('fonts/Roboto/Roboto-MediumItalic.ttf'),
  },
  Montserrat: {
    normal: exports.loadDataFile('fonts/Montserrat/Montserrat-Regular.ttf'),
    bold: exports.loadDataFile('fonts/Montserrat/Montserrat-Bold.ttf'),
    italics: exports.loadDataFile('fonts/Montserrat/Montserrat-Italic.ttf'),
    bolditalics: exports.loadDataFile('fonts/Montserrat/Montserrat-BoldItalic.ttf'),
  },
  MontserratExtraBold: {
    normal: exports.loadDataFile('fonts/Montserrat/Montserrat-ExtraBold.ttf'),
    bold: exports.loadDataFile('fonts/Montserrat/Montserrat-Black.ttf'),
    italics: exports.loadDataFile('fonts/Montserrat/Montserrat-ExtraBoldItalic.ttf'),
    bolditalics: exports.loadDataFile('fonts/Montserrat/Montserrat-BlackItalic.ttf'),
  },
};

// eslint-disable-next-line arrow-body-style
exports.okResponse = (res, httpCode, response) => {
  return res.status(httpCode).json(response);
};

exports.errorResponse = (res, id, extra) => {
  let error = CTS.ERRORS[id];
  error = error || CTS.DEFAULT_ERROR;

  return res.status(error.httpCode).json({
    error: {
      id,
      code: error.code,
      message: error.message,
      extra,
    },
  });
};

exports.sendEmailMailchimp = async params => axios.post(
  'https://mandrillapp.com/api/1.0/messages/send-template.json',
  {
    ...params,
    key: config.get('mailchimp.apiKey'),
  },
  {
    headers: {
      'Content-Type': 'application/json',
    },
  },
);

exports.formatRequestTemplateMailchimp = ({
  templateName: template_name,
  subaccount,
  tags = [],
  email,
  params = [],
}) => {
  const vars = [];
  _.forEach(params, (value, key) => {
    vars.push({ name: key.toUpperCase(), content: value });
  });

  const body = {
    template_name,
    template_content: [{}],
    async: true,
    message: {
      to: [
        {
          email,
        },
      ],
      merge_vars: [
        {
          rcpt: email,
          vars,
        },
      ],
      merge: true,
      merge_language: 'mailchimp',
      subaccount,
      tags,
    },
  };

  return body;
};

exports.updateOrCreate = async (model, where, newItem) => {
  try {
    const foundItem = await model.findOne({ where });
    if (!foundItem) {
      const item = await model.create(newItem);
      return { item, created: true };
    }
    const item = await model.update(newItem, { where });
    return { item, created: false };
  } catch (e) {
    throw e;
  }
};

exports.generatePdf = async data => {
  const { user = {}, suscriptionType = {} } = data;
  const { id: suscriptionTypeId = '' } = suscriptionType;

  const {
    firstName = '', lastName = '', routes = {}, city = '',
  } = user;
  const routeName = routes.length ? routes[0].name : '';
  const fullName = `${firstName} ${lastName}`.replace(/\s+/g, ' ').trim();

  const chartData = await this.calculateQuestionnaire(data);

  let superPower = {};
  let strengthToEnergize = [];
  let sourceDataProcess = [];

  if (suscriptionTypeId === 1) {
    const { cardinalPower = {} } = data;
    superPower = cardinalPower;

    if (chartData.filter(s => s.valueChart >= 3.5 && s.valueChart < 4.5).length) {
      strengthToEnergize = chartData.filter(s => s.valueChart >= 3.5 && s.valueChart < 4.5).pop();
      // console.log('menor de los azules');
    } else if (chartData.filter(s => s.valueChart >= 2 && s.valueChart < 3.5).length) {
      strengthToEnergize = chartData.filter(s => s.valueChart >= 2 && s.valueChart < 3.5).shift();
      // console.log('mayor de los amarillos');
    } else if (chartData.filter(s => s.valueChart < 2).length) {
      strengthToEnergize = chartData.filter(s => s.valueChart < 2).shift();
      // console.log('mayor de los rojos');
    } else {
      strengthToEnergize = chartData.filter(s => s.valueChart >= 4.5).pop();
      // console.log('menor de los veredes');
    }

    const { strength = {} } = strengthToEnergize;

    sourceDataProcess.push({ strength });
  }

  if (suscriptionTypeId === 2 || suscriptionTypeId === 3) {
    let strengthByCardinalPower = _.groupBy(chartData, 'strength.cardinalPowerId');
    strengthByCardinalPower = Object.values(strengthByCardinalPower);
    strengthByCardinalPower = JSON.parse(JSON.stringify(strengthByCardinalPower));

    let acum = 0;
    const cardinalPowers = [];

    strengthByCardinalPower.forEach(x => {
      const totalStrengths = x.length;
      x.forEach(y => {
        const { valueChart = 0 } = y;
        acum += parseFloat(valueChart);
      });
      cardinalPowers.push({
        cardinalPower: x[0].strength.cardinalPower,
        promStrengthByCardinalPower: (acum / totalStrengths).toFixed(2),
      });
      acum = 0;
    });

    // Se el poder cardinal con más puntaje (super poder)
    ({ cardinalPower: [superPower] } = _.sortBy(cardinalPowers, o => o.promStrengthByCardinalPower)
      .reverse()
      .shift());

    sourceDataProcess = chartData;
  }

  // Se obtienen los atributos del super porder
  const { name: superPowerName = '', id: superPowerId = '', description: superPowerDesc } = superPower;

  // Se agregan estilos a la descirpción del super poder
  const superPowerInfo = {
    text: superPowerDesc,
    style: 'superPowerText',
    border: [false, false, false, false],
  };

  const nameStrengthToEnergize = [];
  const strengthToEnergizeText = [];
  const strengthToEnergizeListImplies = [];
  const strengthToEnergizeListRisks = [];
  const brujulaImageName = exports.getBrujulaImage(superPowerId);

  sourceDataProcess.forEach(s => {
    const { strength = {} } = s;
    const { name: auxNameStrengthToEnergize = '' } = strength;
    let { routeStrength = [] } = strength;
    routeStrength = routeStrength.length ? routeStrength[0] : [];
    const { pdfPath = '' } = routeStrength;
    nameStrengthToEnergize.push(auxNameStrengthToEnergize);

    strengthToEnergizeText.push(this.getPdfValue(`${pdfPath}.description`));
    strengthToEnergizeListImplies.push(this.getPdfValue(`${pdfPath}.listOne`));
    strengthToEnergizeListRisks.push(this.getPdfValue(`${pdfPath}.listTwo`));
  });

  // DATA DE ENERGIA LA FORTALEZA
  // return {
  //   superPowerName,
  //   superPowerInfo,
  //   nameStrengthToEnergize,
  //   strengthToEnergizeText,
  //   strengthToEnergizeListImplies,
  //   strengthToEnergizeListRisks,
  // };

  const logoBecrackFooter = {
    image: exports.loadDataFile('pdf-images/logo-text.png'),
    width: 70,
    height: 95,
    alignment: 'right',
    absolutePosition: { x: 400, y: 710 },
    pageBreak: 'after',
  };

  // Se genera sección de energiza tus fortalezas
  const sectionEnergizeStrength = [];
  const strengthMenu = ['\n'];

  nameStrengthToEnergize.forEach((item, i) => {
    sectionEnergizeStrength.push(
      '\n',
      {
        text: [
          {
            text: 'Energiza la fortaleza ',
            alignment: 'center',
            bold: true,
          },
          {
            text: item,
            alignment: 'center',
            bold: true,
            style: ['colorBlue'],
            id: `list${i}`,
          },
        ],
      },
      '\n\n',
      {
        text: strengthToEnergizeText[i],
        alignment: 'center',
      },
      '\n\n',
      {
        image: exports.loadDataFile('pdf-images/image-implica.png'),
        width: 90,
        height: 80,
        alignment: 'center',
      },
      '\n\n',
      {
        text: [
          {
            text: 'Energizar tu ',
            alignment: 'center',
            bold: true,
          },
          {
            text: `“${item}”`,
            alignment: 'center',
            bold: true,
            italics: true,
          },
          {
            text: ' implica:',
            alignment: 'center',
            bold: true,
          },
        ],
      },
      '\n\n',
      strengthToEnergizeListImplies[i],
      logoBecrackFooter,
      {
        image: exports.loadDataFile('pdf-images/image-riesgos.png'),
        width: 90,
        height: 80,
        alignment: 'center',
      },
      '\n\n',
      {
        text: [
          {
            text: 'Cuando tu fortaleza ',
            alignment: 'center',
            bold: true,
          },
          {
            text: `“${item}”`,
            alignment: 'center',
            bold: true,
            italics: true,
          },
          {
            text: ' está apagada, tienes riesgos:',
            alignment: 'center',
            bold: true,
          },
        ],
      },
      '\n\n',
      strengthToEnergizeListRisks[i],
      logoBecrackFooter,
    );
    strengthMenu.push(
      {
        text: item,
        alignment: 'center',
        decoration: 'underline',
        decorationColor: 'black',
        marginTop: 10,
        marginBottom: 10,
        linkToDestination: `list${i}`,
      },
    );
  });

  sectionEnergizeStrength.push(
    {
      text: [
        {
          text: '¿',
          style: ['fontSize22', 'colorBlue'],
        },
        { text: 'Cómo continuar', id: 'lastPage' },
        {
          text: '?',
          style: ['fontSize22', 'colorBlue'],
        },
      ],
      bold: true,
    },
    '\n',
    {
      text: [
        {
          text: 'Activa tu ',
        },
        { text: 'Plan de Coaching', bold: true, italics: true },
        {
          text: ` para crecer en tu RUTA ${routeName.toUpperCase()}.`,
        },
      ],
    },
    '\n',
    {
      ul: [
        {
          text: [
            { text: 'Completa el cuestionario y conócete 100% apreciando todas tus ' },
            { text: 'fortalezas', bold: true },
          ],
        },
        {
          text: [
            'Accede a una ruta que te guiará en tus retos de ',
            { text: 'emprendimiento', bold: true },
            ', aprovechando tus fortalezas en ventaja y mejorando las que están en riesgo.',
          ],
        },
        {
          text: [
            'Haz seguimiento a tus resultados, escucha lo que la comunidad de Cracks tiene para ti, y crea hábitos que te llevarán a un siguiente nivel como persona y como ',
            { text: 'líder', bold: true },
            '.',
          ],
        },
      ],
    },
    '\n\n',
    {
      image: exports.loadDataFile('pdf-images/btn-plan-coaching.png'),
      width: 150,
      height: 35,
      alignment: 'center',
    },
    '\n',
    {
      image: exports.loadDataFile('pdf-images/btn-informe-completo.png'),
      width: 150,
      height: 35,
      alignment: 'center',
    },
  );

  try {
    const chart = await exports.getChartReport(chartData);
    // exports.loadDataFile('pdf-images/imagen-eje-x.png');

    const content = [];

    /** ******************* portada ******************************************* */
    content.push(
      {
        text: 'CONÓCETE PARA CAMBIAR MEJOR',
        alignment: 'center',
        characterSpacing: 1.5,
        style: 'fontSize16',
      },
      '\n\n',
      {
        text: 'INFORME DE PODERES Y FORTALEZAS',
        alignment: 'center',
        bold: true,
        style: 'fontSize22',
      },
      '\n',
      {
        text: `${fullName}`,
        style: 'userName',
      },
      {
        text: `${city} ${moment().format('DD-MM-YYYY')}`,
        style: 'cityDate',
      },
      '\n\n',
      {
        text: 'Tus fortalezas son el combustible para despegar hacia un nuevo objetivo.',
        alignment: 'center',
      },
      {
        text: [
          {
            text:
              'Es inteligente tomar un tiempo para conocerte pues el recurso más valioso con el que cuentas eres ',
            alignment: 'center',
          },
          { text: 'tú.', bold: true },
        ],
      },
      '\n\n',
      {
        image: exports.loadDataFile('pdf-images/logo-portada.png'),
        width: 650,
        height: 180,
        alignment: 'center',
      },
      '\n\n\n',
      {
        text: [
          {
            text: 'Tienes Poder para',
            alignment: 'center',
            bold: true,
            style: 'fontSize22',
          },
          {
            text: superPowerName.replace('Poder para', ''),
            alignment: 'center',
            bold: true,
            style: ['fontSize22', 'colorBlue'],
          },
        ],
      },
      '\n\n',
      {
        text: 'BeCrack ha creado un modelo para ayudarte a descubrir los ',
        alignment: 'center',
      },
      {
        text: 'rasgos significativos de tu liderazgo; éstos se agrupan en 4 poderes cardinales:',
        alignment: 'center',
      },
      {
        text:
          'Poder para Crear, Poder para Transformar, Poder para Descubrir y Poder para Influir en otros.',
        alignment: 'center',
        margin: [65, 0, 65, 0],
      },
      '\n\n',
      logoBecrackFooter,
    );
    /** ******************* pagina 2 ******************************************* */
    content.push(
      {
        margin: [-40, 0, 0, -10],
        table: {
          widths: ['108%'],
          body: [
            [
              {
                text: superPowerName,
                style: 'superPowerTitle',
                border: [false, false, false, false],
              },
            ],
            [superPowerInfo],
          ],
        },
      },
      {
        image: exports.loadDataFile('pdf-images/btn-becrack.png'),
        width: 100,
        height: 25,
        alignment: 'center',
      },
      '\n\n',
      {
        image: exports.loadDataFile('pdf-images/logo-tus-fortalezas.png'),
        width: 100,
        height: 100,
        alignment: 'center',
      },
      {
        text: 'Tus fortalezas',
        style: 'fontSize22',
        alignment: 'center',
        bold: true,
      },
      '\n\n',
      {
        text: 'Las fortalezas se expresan en tus comportamientos,',
        alignment: 'center',
      },
      {
        text: 'lenguaje y decisiones son recursos que puedes potenciar de acuerdo',
        alignment: 'center',
      },
      {
        text: 'con los objetivos que te planteas. ¿Qué fortaleza necesitas encender',
        alignment: 'center',
      },
      {
        text: 'para crear tu propio negocio? ¿Cuáles son la clave para crecer en el mundo',
        alignment: 'center',
      },
      {
        text: [
          {
            text: 'profesional o para reinventar tu estilo de vida? ',
            alignment: 'center',
          },
          { text: 'BeCrack ', bold: true },
          { text: 'te acompaña' },
        ],
      },
      {
        text: 'a encontrar respuestas.',
        alignment: 'center',
      },
      {
        text: 'En este informe verás las fortalezas del súper poder que elegiste,',
        alignment: 'center',
      },
      {
        text: 'resultado de las respuestas que seleccionaste en el cuestionario de fortalezas.',
        alignment: 'center',
      },
      logoBecrackFooter,
    );

    /** ******************* pagina 3 ******************************************* */
    content.push(
      {
        text: 'BeCrack te muestra los resultados del cuestionario de valoración de fortalezas.',
        alignment: 'center',
      },
      {
        text: 'Procesamos tus respuestas usando nuestro modelo de fortalezas para innovar.',
        alignment: 'center',
      },
      '\n',
      {
        text: '¿Cómo usar este informe?',
        alignment: 'center',
      },
      '\n',
      {
        text: '1. Observa los gráficos para tener una visión general de tus capacidades:',
        alignment: 'center',
      },
    );

    if (suscriptionTypeId === 1) {
      content.push(
        {
          text: '¿Cuáles son tus ventajas? ¿Cuáles son tus fallas o tus riegos?',
          alignment: 'center',
        },
        '\n\n\n\n\n',
        {
          image: chart,
          width: 400,
          style: 'imageChart',
        },
        {
          image: exports.loadDataFile('pdf-images/imagen-eje-x.png'),
          width: 343,
          absolutePosition: { x: 155, y: 430 },
        },
        '\n\n\n\n\n',
        {
          text:
            'Este gráfico te permite descubrir la intensidad con que se presentan tus fortalezas.',
          alignment: 'center',
        },
        logoBecrackFooter,
        {
          text:
            '2. Da los primeros pasos hacia tu objetivo usando la guía para energizar fortalezas y evaluando tus riesgos y acciones',
          alignment: 'center',
        },
        '\n\n',
        {
          image: exports.loadDataFile('pdf-images/guia-energizar.png'),
          width: 400,
          height: 350,
          alignment: 'center',
        },
        '\n\n',
        {
          text: 'Sugerencia de BeCrack',
          alignment: 'center',
        },
        {
          text: `Energiza la fortaleza: ${nameStrengthToEnergize}`,
          alignment: 'center',
        },
        logoBecrackFooter,
        {
          text:
            '3. Aprende más sobre las acciones para energizar tus fortalezas y comienza a tomar medidas. Más adelante encontrarás recomendaciones para las competencias que requieres en tu reto específico.',
          alignment: 'center',
        },
        '\n\n',
        {
          image: exports.loadDataFile('pdf-images/recomendaciones.png'),
          width: 400,
          height: 400,
          alignment: 'center',
        },
        logoBecrackFooter,
        {
          text: [
            {
              text: 'Sugerencia de ',
              alignment: 'center',
              bold: true,
              style: 'fontSize22',
            },
            {
              text: 'BeCrack',
              alignment: 'center',
              bold: true,
              style: ['fontSize22', 'colorBlue'],
            },
          ],
        },
        '\n',
        {
          text: `Energiza la fortaleza: ${nameStrengthToEnergize}`,
          alignment: 'center',
          bold: true,
        },
        '\n\n',
        strengthToEnergizeText,
        logoBecrackFooter,
        {
          image: exports.loadDataFile('pdf-images/image-riesgos.png'),
          width: 90,
          height: 80,
          alignment: 'center',
        },
        '\n\n',
        {
          text: [
            {
              text: 'Cuando tu fortaleza',
            },
            { text: ` “${nameStrengthToEnergize}” `, italics: true },
            {
              text: ' está apagada, tienes riesgos:',
            },
          ],
          bold: true,
          // alignment: 'center',
        },
        '\n\n',
        strengthToEnergizeListRisks,
        '\n\n',
        {
          text: [
            {
              text: '¿',
              style: ['fontSize22', 'colorBlue'],
            },
            { text: 'Cómo continuar', id: 'lastPage' },
            {
              text: '?',
              style: ['fontSize22', 'colorBlue'],
            },
          ],
          bold: true,
        },
        '\n',
        {
          text: [
            {
              text: 'Activa tu ',
            },
            { text: 'Plan de Coaching', bold: true, italics: true },
            {
              text: ` para crecer en tu RUTA ${routeName.toUpperCase()}.`,
            },
          ],
        },
        '\n',
        {
          ul: [
            {
              text: [
                { text: 'Completa el cuestionario y conócete 100% apreciando todas tus ' },
                { text: 'fortalezas', bold: true },
              ],
              // leadingIndent: 30,
            },
            {
              text: [
                'Accede a una ruta que te guiará en tus retos de ',
                { text: 'emprendimiento', bold: true },
                ', aprovechando tus fortalezas en ventaja y mejorando las que están en riesgo.',
              ],
            },
            {
              text: [
                'Haz seguimiento a tus resultados, escucha lo que la comunidad de Cracks tiene para ti, y crea hábitos que te llevarán a un siguiente nivel como persona y como ',
                { text: 'líder', bold: true },
                '.',
              ],
            },
          ],
          // lineHeight: 1,
        },
        '\n\n',
        {
          image: exports.loadDataFile('pdf-images/btn-plan-coaching.png'),
          width: 150,
          height: 35,
          alignment: 'center',
        },
        '\n',
        {
          image: exports.loadDataFile('pdf-images/btn-informe-completo.png'),
          width: 150,
          height: 35,
          alignment: 'center',
        },
        logoBecrackFooter,
        {
          image: exports.loadDataFile('pdf-images/image-implica.png'),
          width: 90,
          height: 80,
          alignment: 'center',
        },
        '\n\n',
        {
          text: [
            {
              text: 'Energizar tu',
            },
            { text: ` “${nameStrengthToEnergize}” `, italics: true },
            {
              text: 'implica',
            },
          ],
          bold: true,
          alignment: 'center',
        },
        '\n\n',
        strengthToEnergizeListImplies,
        {
          image: exports.loadDataFile('pdf-images/logo-text.png'),
          width: 70,
          height: 95,
          alignment: 'right',
          absolutePosition: { x: 400, y: 710 },
        },
      );
    } else if (suscriptionTypeId === 2) {
      content.push(
        {
          text: '¿Cómo es el equilibrio entre tus poderes y fortalezas? ¿Cuáles son tus ventajas? ',
          alignment: 'center',
        },
        '\n\n\n',
      );

      const dataToCompassChart = _.sortBy(chartData, o => o.strength.cardinalPowerId);
      const chartCompass = await exports.getChartCompass(dataToCompassChart);
      // exports.loadDataFile('pdf-images/imagen-eje-x.png');

      content.push(
        {
          image: chartCompass,
          width: 400,
          style: 'imageChart',
        },
        {
          image: exports.loadDataFile(`pdf-images/${brujulaImageName}`),
          width: 403,
          height: 276,
          absolutePosition: { x: 92, y: 202 },
        },
        '\n',
        {
          image: chart,
          width: 400,
          style: 'imageChart',
        },
        {
          image: exports.loadDataFile('pdf-images/imagen-eje-x.png'),
          width: 332,
          absolutePosition: { x: 166, y: 682 },
        },
        logoBecrackFooter,
        {
          text: 'Sugerencia de BeCrack',
          alignment: 'center',
          bold: true,
        },
        {
          text: [
            {
              text: 'Ruta ',
              alignment: 'center',
              bold: true,
            },
            {
              text: routeName,
              alignment: 'center',
              bold: true,
              style: 'colorBlue',
            },
          ],
        },
        '\n\n',
        {
          text: 'La experiencia exitosa de otros Cracks en la ruta Emprender',
          alignment: 'center',
        },
        {
          text: 'nos inspira a proponerte Energizar las siguientes fortalezas.',
          alignment: 'center',
        },
        '\n\n',
        {
          text: nameStrengthToEnergize.toString().replace(/,/g, ' - '),
          alignment: 'center',
          bold: true,
        },
        '\n\n',
        {
          text: 'También puedes hacer la elección que consideres adecuada para el momento que vives y seleccionar en el menú de fortalezas.',
          alignment: 'center',
        },
        '\n\n\n\n',
        {
          image: exports.loadDataFile('pdf-images/menu-fortalezas.png'),
          width: 180,
          height: 40,
          absolutePosition: { x: 205, y: 317 },
        },
        {
          margin: [120, 0, 0, 0],
          fillColor: '#e0e0e0',
          layout: 'noBorders',
          table: {
            widths: ['70%'],
            body: [
              [
                strengthMenu,
              ],
            ],
          },
        },
        logoBecrackFooter,
        {
          text:
            '2. Da los primeros pasos hacia tu objetivo usando la guía para energizar fortalezas y evaluando tus riesgos y acciones',
          alignment: 'center',
        },
        '\n\n',
        {
          image: exports.loadDataFile('pdf-images/guia-energizar.png'),
          width: 400,
          height: 350,
          alignment: 'center',
        },
        '\n\n',
        {
          text: 'Sugerencia de BeCrack',
          alignment: 'center',
        },
        {
          text: `Energiza la fortaleza: ${nameStrengthToEnergize.toString().replace(/,/g, ' , ')}`,
          alignment: 'center',
        },
        logoBecrackFooter,
        {
          text:
            '3. Aprende más sobre las acciones para energizar tus fortalezas y comienza a tomar medidas. Más adelante encontrarás recomendaciones para las competencias que requieres en tu reto específico.',
          alignment: 'center',
        },
        '\n\n',
        {
          image: exports.loadDataFile('pdf-images/recomendaciones.png'),
          width: 400,
          height: 400,
          alignment: 'center',
        },
        logoBecrackFooter,
        sectionEnergizeStrength,
        {
          image: exports.loadDataFile('pdf-images/logo-text.png'),
          width: 70,
          height: 95,
          alignment: 'right',
          absolutePosition: { x: 400, y: 710 },
        },
      );
    } else if (suscriptionTypeId === 3) {
      content.push(

        {
          text: '¿Cómo es el equilibrio entre tus poderes y fortalezas? ¿Cuáles son tus ventajas? ',
          alignment: 'center',
        },
        '\n\n\n',
      );

      const dataToCompassChart = _.sortBy(chartData, o => o.strength.cardinalPowerId);
      const chartCompass = await exports.getChartCompass(dataToCompassChart);
      // exports.loadDataFile('pdf-images/imagen-eje-x.png');

      content.push(
        {
          image: chartCompass,
          width: 400,
          style: 'imageChart',
        },
        {
          image: exports.loadDataFile(`pdf-images/${brujulaImageName}`),
          width: 403,
          height: 276,
          absolutePosition: { x: 92, y: 202 },
        },
        '\n',
        {
          image: chart,
          width: 400,
          style: 'imageChart',
        },
        {
          image: exports.loadDataFile('pdf-images/imagen-eje-x.png'),
          width: 332,
          absolutePosition: { x: 166, y: 682 },
        },
        logoBecrackFooter,
        {
          text: 'Tu Plan de Coaching',
          alignment: 'center',
          bold: true,
        },
        {
          text: [
            {
              text: 'Ruta ',
              alignment: 'center',
              bold: true,
            },
            {
              text: routeName,
              alignment: 'center',
              bold: true,
              style: 'colorBlue',
            },
          ],
        },
        '\n\n\n',
        {
          text: 'BeCrack te acompañará a fortalecer tus capacidades para Emprender guiándote a través de diferentes retos que tendrás la oportunidad de seleccionar. En cada uno, podrás sintonizarte con tus objetivos y las energizar las fortalezas más adecuadas para tu proceso específico.',
          alignment: 'center',
        },
        '\n\n',
        {
          text:
            '2. Da los primeros pasos hacia tu objetivo usando la guía para energizar fortalezas y evaluando tus riesgos y acciones',
          alignment: 'center',
        },
        '\n',
        {
          image: exports.loadDataFile('pdf-images/guia-energizar.png'),
          width: 400,
          height: 350,
          alignment: 'center',
        },
        '\n\n',
        {
          text: 'Sugerencia de BeCrack',
          alignment: 'center',
        },
        {
          text: `Energiza la fortaleza: ${nameStrengthToEnergize.toString().replace(/,/g, ' , ')}`,
          alignment: 'center',
        },
        logoBecrackFooter,
        {
          text:
            '3. Aprende más sobre las acciones para energizar tus fortalezas y comienza a tomar medidas. Más adelante encontrarás recomendaciones para las competencias que requieres en tu reto específico.',
          alignment: 'center',
        },
        '\n\n',
        {
          image: exports.loadDataFile('pdf-images/recomendaciones.png'),
          width: 400,
          height: 400,
          alignment: 'center',
        },
        {
          image: exports.loadDataFile('pdf-images/logo-text.png'),
          width: 70,
          height: 95,
          alignment: 'right',
          absolutePosition: { x: 400, y: 710 },
        },
      );
    }

    const docDefinition = {
      pageMargins: [40, 60, 40, 60],
      content,
      defaultStyle: {
        font: 'Montserrat',
      },
      styles: {
        colorBlue: {
          color: '#00bdf7',
        },
        fontSize22: {
          fontSize: 22,
        },
        fontSize16: {
          fontSize: 16,
        },
        userName: {
          font: 'MontserratExtraBold',
          alignment: 'center',
          fontSize: 22,
          color: '#00bdf7',
        },
        cityDate: {
          alignment: 'center',
          fontSize: 16,
          bold: true,
        },
        superPowerTitle: {
          alignment: 'center',
          fillColor: '#e0e0e0',
          marginTop: 10,
          bold: true,
        },
        superPowerText: {
          alignment: 'center',
          fillColor: '#e0e0e0',
          margin: [60, 0, 60, 10],
          bold: true,
        },
        moreInfoLink: {
          alignment: 'center',
          color: 'blue',
          decoration: 'underline',
          decorationColor: 'blue',
        },
        imageChart: {
          alignment: 'center',
        },
      },
    };

    const printer = new PdfPrinter(fonts);

    const pdfDoc = printer.createPdfKitDocument(docDefinition);
    pdfDoc.end();
    const fileName = `informe-${moment().format('YYYY-MM-DD-hh-mm-ss')}.pdf`;
    // Para crear archivo físico
    // pdfDoc.pipe(fs.createWriteStream(path.resolve(__dirname, fileName)));
    await awsActions.uploadFile(fileName, pdfDoc);
    const urlPdf = await awsActions.getSignedUrl(fileName);
    return urlPdf;
  } catch (e) {
    throw e;
  }
};

exports.getChartCompass = async data => {
  const categories = [];
  const auxData = [];
  data.forEach(x => {
    const { strength = {}, valueChart = 0 } = x;
    const { name = '' } = strength;
    categories.push(name);
    auxData.push({
      y: parseFloat(valueChart),
      color: exports.getRangeColorFromValue(valueChart),
    });
  });

  try {
    const exportSettings = {
      scale: 2,
      options: {
        chart: {
          polar: true,
          type: 'line',
          margin: 50,
        },
        title: {
          text: null,
        },
        pane: {
          size: '100%',
          startAngle: 15,
        },
        xAxis: {
          categories,
          lineWidth: 1,
          labels: {
            style: {
              overflow: 'visible',
            },
          },
        },

        yAxis: {
          gridLineInterpolation: 'circle',
          lineWidth: 0,
          min: 0,
          max: 6,
          tickPositions: [0, 1, 2, 3, 4, 5],
        },
        credits: {
          enabled: false,
        },
        series: [
          {
            showInLegend: false,
            name: '',
            data: auxData,
            pointPlacement: 'on',
          },
        ],
      },
    };

    const { data: resultExternalChart } = await exports.getExternalReportChart({ exportSettings });

    const { chartBase64 = '' } = resultExternalChart;
    return chartBase64;
  } catch (e) {
    throw e;
  }
};

exports.getChartReport = async data => {
  const categories = [];
  const auxData = [];
  data.forEach(x => {
    const { strength = {}, valueChart = 0 } = x;
    const { name = '' } = strength;
    categories.push(name);
    auxData.push({
      y: parseFloat(valueChart),
      color: exports.getRangeColorFromValue(valueChart),
    });
  });

  try {
    const exportSettings = {
      scale: 3,
      options: {
        chart: {
          type: 'bar',
          height: 300,
          plotBackgroundColor: '#ddd',
        },
        title: {
          text: null,
        },
        xAxis: {
          categories,
          title: {
            text: null,
          },
          gridLineColor: '#ffffff',
          gridLineWidth: 2,
        },
        yAxis: {
          min: 0,
          max: 6,
          title: {
            align: 'high',
            text: null,
          },
          labels: {
            overflow: 'justify',
            format: '{value:.2f}',
            enabled: false,
          },
          gridLineColor: '#ffffff',
          gridLineWidth: 2,
        },
        plotOptions: {
          bar: {
            dataLabels: {
              enabled: true,
            },
            borderRadius: 5,
          },
        },
        credits: {
          enabled: false,
        },
        series: [
          {
            showInLegend: false,
            name: 'asd',
            data: auxData,
          },
        ],
      },
    };
    const { data: resultExternalChart } = await exports.getExternalReportChart({ exportSettings });

    const { chartBase64 = '' } = resultExternalChart;
    return chartBase64;
  } catch (error) {
    throw error;
  }
};

exports.getRangeColorFromValue = value => {
  let color = '';
  if (value < 2) {
    color = '#e5152e';
  } else if (value < 4) {
    color = '#f2cd00';
  } else if (value < 5) {
    color = '#00bdf7';
  } else {
    color = '#66e515';
  }

  return color;
};

exports.calculateQuestionnaire = async (data, round) => {
  const { behavior = [] } = data;
  let finalResponse = [];
  let acumBehaviorPoints = 0;

  let behaviorsByStrength = _.groupBy(behavior, 'strengthId');
  behaviorsByStrength = Object.values(behaviorsByStrength);
  behaviorsByStrength = JSON.parse(JSON.stringify(behaviorsByStrength));
  behaviorsByStrength.forEach(x => {
    const totalBehavior = x.length;
    x.forEach(y => {
      const { behaviorPoint = [] } = y;
      acumBehaviorPoints += behaviorPoint.length ? parseFloat(behaviorPoint[0].weight) : 0;
    });
    finalResponse.push({
      strength: {
        ...x[0].strength,
        id: x[0].strengthId,
      },
      valueChart: (acumBehaviorPoints / totalBehavior).toFixed(2),
      percent: round
        ? Math.round((acumBehaviorPoints / totalBehavior) * 20)
        : ((acumBehaviorPoints / totalBehavior) * 20).toFixed(2),
    });
    acumBehaviorPoints = 0;
  });

  finalResponse = _.sortBy(finalResponse, o => o.valueChart).reverse();
  return finalResponse;
};

exports.getPdfValue = jsonPath => {
  const res = jsonPath.split('.').reduce(
    (o, k) => o && o[k],
    pdfTexts.texts,
  );

  return res;
};

exports.getMobileValue = jsonPath => {
  const res = jsonPath.split('.').reduce(
    (o, k) => o && o[k],
    appText.texts,
  );

  return res;
};

exports.getBrujulaImage = superPowerId => {
  switch (superPowerId) {
    case 1: {
      return 'brujula-crear.png';
    }
    case 2: {
      return 'brujula-transformar.png';
    }
    case 3: {
      return 'brujula-descubrir.png';
    }
    case 4: {
      return 'brujula-influir.png';
    }
    default: {
      return 'brujula-crear.png';
    }
  }
};

exports.getExternalReportChart = async params => axios.post(
  'http://ec2-3-88-220-190.compute-1.amazonaws.com:4545/api/reports/chart',
  params,
  {
    headers: {
      'Content-Type': 'application/json',
    },
  },
);
