const convict = require('convict');
const fs = require('fs');

// Define a schema
const config = convict({
  env: {
    doc: 'The application environment.',
    format: ['prod', 'dev'],
    default: 'dev',
    env: 'NODE_ENV',
  },
  db: {
    host: {
      doc: 'Database host name/IP',
      format: String,
      default: '',
      env: 'DB_HOST',
    },
    name: {
      doc: 'Database name',
      format: String,
      default: '',
      env: 'DB_NAME',
    },
    user: {
      doc: 'The db username.',
      format: String,
      default: '',
      env: 'DB_USER',
    },
    password: {
      doc: 'The db password.',
      format: String,
      default: '',
      env: 'DB_PASSWORD',
    },
    socketPath: {
      doc: 'The db socket path.',
      format: String,
      default: '',
      env: 'DB_SOCKET_PATH',
    },
    port: {
      doc: 'The db port.',
      format: 'port',
      default: '',
      env: 'DB_PORT',
    },
    dialect: {
      doc: 'The dialect of the DBMS.',
      format: String,
      default: '',
      env: 'DB_DIALECT',
    },
    pool: {
      max: {
        doc: 'The max pool connection to DBMS.',
        format: 'int',
        default: '',
        env: 'DB_MAX_CONNECTION',
      },
      min: {
        doc: 'The min pool connection to DBMS.',
        format: 'int',
        default: '',
        env: 'DB_MIN_CONNECTION',
      },
      acquire: {
        doc: 'The acquire pool connection to DBMS.',
        format: 'int',
        default: '',
        env: 'DB_ACQUIRE',
      },
      idle: {
        doc: 'The idle pool connection to DBMS.',
        format: 'int',
        default: '',
        env: 'DB_IDLE',
      },
    },
  },
  jwt: {
    secret: {
      doc: 'The secret word to create token.',
      format: String,
      default: '',
      env: 'JWT_SECRET',
    },
    tokenExpirationTime: {
      doc: 'The time expiration of token.',
      format: 'int',
      default: '',
      env: 'JWT_TOKEN_EXPIRATION_TIME',
    },
  },
  vpc: {
    subnets: {
      doc: 'The subnets vpc.',
      format: String,
      default: '',
      env: 'VPC_SUBNETS',
    },
    securityGroups: {
      doc: 'The security groups vpc.',
      format: String,
      default: '',
      env: 'VPC_SECURITY_GROUPS',
    },
  },
  cors: {
    urls: {
      doc: 'The urls access cors.',
      format: String,
      default: '',
      env: 'CORS_URLS',
    },
  },
  validationCodes: {
    expirationTimeCodeEmail: {
      doc: 'Expiration time of the mail validation code (min).',
      format: 'int',
      default: '',
      env: 'EXPIRATION_TIME_CODE_EMAIL',
    },
    expirationTimeCodeResetPassword: {
      doc: 'expiration time of the validation code for password change (min).',
      format: 'int',
      default: '',
      env: 'EXPIRATION_TIME_CODE_RESET_PASSWORD',
    },
  },
  mailchimp: {
    apiKey: {
      doc: 'The mailchimp api key.',
      format: String,
      default: '',
      env: 'MAILCHIMP_API_KEY',
    },
  },
  aws: {
    bucket: {
      doc: 'The bucket name aws s3.',
      format: String,
      default: '',
      env: 'S3_BUCKET',
    },
    accessKey: {
      doc: 'The access key aws.',
      format: String,
      default: '',
      env: 'ACCESS_KEY_ID_AWS',
    },
    secretAccessKey: {
      doc: 'The secret access key aws.',
      format: String,
      default: '',
      env: 'SECRET_ACCESS_KEY_AWS',
    },
    region: {
      doc: 'The region key aws.',
      format: String,
      default: '',
      env: 'REGION_AWS',
    },
    platformAppAndroid: {
      doc: 'The platorm app arn android sns aws.',
      format: String,
      default: '',
      env: 'PLATFORM_APP_ANDROID',
    },
    platformAppIos: {
      doc: 'The platorm app arn ios sns aws.',
      format: String,
      default: '',
      env: 'PLATFORM_APP_IOS',
    },
  },
});

const env = config.get('env');
const envFile = `./env/${env}.json`;

if (fs.existsSync(envFile)) config.loadFile(envFile);

// Perform validation
config.validate({ allowed: 'warn' });

module.exports = config;
